<?php

namespace App\Http\Controllers\Admin;

use Setting;
use Illuminate\Support\Str;
use Illuminate\Http\Request;
use App\Helpers\Classes\ArtisanApi;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Session;

class UpdateController extends Controller
{
    protected $manager;
    protected $version;

    public function __construct(ArtisanApi $manager)
    {
        $this->manager = $manager;
        $this->version = setting('version');
    }

    public function show()
    {
        return view('update');
    }

    public function patches()
    {
        // Since we are no longer fetching patches from external APIs,
        // you can either load patches manually from a local store or leave this empty.
        $version = setting('version');
        $applied = setting("patches-" . Str::slug($version) . "-applied", '{}');
        $applied = json_decode($applied, true);

        // Load manually defined patches if needed
        $patches = [];  // No external patch fetching now

        return view('patches', compact('patches', 'applied'));
    }

    public function applyPatches(Request $request, $id)
    {
        // Since patches are no longer fetched externally, this part will apply local patches.
        // Assuming patches are manually provided:
        $patches = Cache::get('patches-available', []);
        $download = collect($patches)
            ->filter(function ($patch) use ($id) {
                return $id == $patch['id'];
            })
            ->map(function ($patch) {
                return $patch['download'];  // This download URL can now be a local server path
            })
            ->first();

        // If download is found and patch is applied successfully
        if ($download && $this->manager->downloadPatch($download)) {
            $applied = setting("patches-" . Str::slug($this->version) . "-applied", '{}');
            $applied = json_decode($applied, true);

            // Save applied patch
            if (!in_array($id, $applied)) {
                $applied[] = $id;
                Setting::set("patches-" . Str::slug($this->version) . "-applied", json_encode($applied));
                Setting::save();
            }
        }

        return redirect()->route('system.patches');
    }

    public function update()
    {
        // Run the update (local update now, no external check)
        $messages = $this->manager->runUpdate();

        // Set success message in settings
        Setting::set('success-message', 'Application updated successfully!');
        Setting::save();

        return to_route('admin.dashboard');
    }

    public function checkUpdates()
    {
        // Since we no longer check for updates externally, just mark updates as checked.
        $this->manager->checkUpdates();

        return redirect()->route('admin.dashboard');
    }

    public function verifyUpdates()
    {
        // Perform local update validation using the default code
        if ($this->manager->validateAndPerformUpdate()) {
            return redirect()->route('system.update');
        }
    
        return redirect()->route('admin.dashboard');
    }

}
